<?php

namespace App\Services;

use App\Jobs\SendCourierPushReminder;
use App\Models\Delivery;
use App\Models\Courier;
use App\Models\DeliveryRejection;
use Illuminate\Support\Carbon;
use App\Services\PushNotificationService;
use App\Services\TelegramNotificationService;

class DeliveryAssigner
{
    protected function courierHasCapacity(Courier $courier): bool
    {
        $maxActive = $courier->users()->value('max_active_orders') ?? 3;
        $activeCount = Delivery::query()
            ->where('courier_id', $courier->id)
            ->whereIn('status', Delivery::ACTIVE_STATUSES)
            ->count();

        return $activeCount < $maxActive;
    }

    public function assignNext(Delivery $delivery): ?Courier
    {
        $restaurant = $delivery->restaurant;
        $originLat = $restaurant?->lat ?? $restaurant?->zone?->center_lat;
        $originLng = $restaurant?->lng ?? $restaurant?->zone?->center_lng;

        $hasOrigin = $originLat && $originLng;

        $rejectedCourierIds = DeliveryRejection::where('delivery_id', $delivery->id)
            ->pluck('courier_id')
            ->toArray();

        $query = Courier::query()
            ->where('is_active', true)
            ->where('is_online', true);

        if ($hasOrigin) {
            $query->whereNotNull('last_lat')
                ->whereNotNull('last_lng')
                ->where('last_seen_at', '>=', Carbon::now()->subMinutes(60));
        }

        if ($restaurant?->zone_id) {
            $query->where('zone_id', $restaurant->zone_id);
        }

        if (!empty($rejectedCourierIds)) {
            $query->whereNotIn('id', $rejectedCourierIds);
        }

        if ($hasOrigin) {
            $query->orderByRaw(
                '(6371 * acos(cos(radians(?)) * cos(radians(last_lat)) * cos(radians(last_lng) - radians(?)) + sin(radians(?)) * sin(radians(last_lat)))) asc',
                [$originLat, $originLng, $originLat]
            );
        }

        $couriers = $query->limit(20)->get();
        $courier = null;
        foreach ($couriers as $candidate) {
            if ($this->courierHasCapacity($candidate)) {
                $courier = $candidate;
                break;
            }
        }

        if (!$courier) {
            $fallbackQuery = Courier::query()
                ->where('is_active', true)
                ->where('is_online', true);
            if ($restaurant?->zone_id) {
                $fallbackQuery->where('zone_id', $restaurant->zone_id);
            }
            if (!empty($rejectedCourierIds)) {
                $fallbackQuery->whereNotIn('id', $rejectedCourierIds);
            }
            $fallbackCandidates = $fallbackQuery->orderBy('id')->limit(20)->get();
            foreach ($fallbackCandidates as $candidate) {
                if ($this->courierHasCapacity($candidate)) {
                    $courier = $candidate;
                    break;
                }
            }
        }
        
        if (!$courier) {
            $moreCandidates = Courier::query()
                ->where('is_active', true)
                ->where('is_online', true)
                ->when(!empty($rejectedCourierIds), fn($q) => $q->whereNotIn('id', $rejectedCourierIds))
                ->orderBy('id')
                ->limit(20)
                ->get();
            foreach ($moreCandidates as $candidate) {
                if ($this->courierHasCapacity($candidate)) {
                    $courier = $candidate;
                    break;
                }
            }
        }

        if (!$courier) {
            $hasOtherPending = Delivery::query()
                ->where('status', Delivery::STATUS_PENDING)
                ->whereNull('courier_id')
                ->where('id', '!=', $delivery->id)
                ->exists();

            if (!$hasOtherPending) {
                DeliveryRejection::where('delivery_id', $delivery->id)->delete();
                $finalCandidates = Courier::query()
                    ->where('is_active', true)
                    ->where('is_online', true)
                    ->orderBy('id')
                    ->limit(20)
                    ->get();
                foreach ($finalCandidates as $candidate) {
                    if ($this->courierHasCapacity($candidate)) {
                        $courier = $candidate;
                        break;
                    }
                }
            }
        }

        if ($courier) {
            $delivery->update([
                'courier_id' => $courier->id,
                'status' => Delivery::STATUS_PENDING,
                'assigned_at' => now(),
                'push_attempt' => 0,
                'push_last_sent_at' => null,
            ]);
            $this->notifyCourier($courier, $delivery);
        } else {
            $delivery->update([
                'courier_id' => null,
                'status' => Delivery::STATUS_PENDING,
            ]);
        }

        return $courier;
    }

    public function assignNextForCourier(int $courierId, ?int $excludeDeliveryId = null): ?Delivery
    {
        $courier = Courier::find($courierId);
        if (!$courier) {
            return null;
        }
        if (!$courier->is_active || !$courier->is_online) {
            return null;
        }
        if (!$this->courierHasCapacity($courier)) {
            return null;
        }

        $query = Delivery::query()
            ->where('status', Delivery::STATUS_PENDING)
            ->whereNull('courier_id');

        if ($excludeDeliveryId) {
            $query->where('id', '!=', $excludeDeliveryId);
        }

        if ($courier->zone_id) {
            $query->whereHas('restaurant', function ($subQuery) use ($courier) {
                $subQuery->where('zone_id', $courier->zone_id);
            });
        }

        $delivery = $query->orderBy('created_at')->first();

        if (!$delivery && $excludeDeliveryId) {
            $delivery = Delivery::query()
                ->where('status', Delivery::STATUS_PENDING)
                ->whereNull('courier_id')
                ->when($courier->zone_id, function ($subQuery) use ($courier) {
                    $subQuery->whereHas('restaurant', function ($zoneQuery) use ($courier) {
                        $zoneQuery->where('zone_id', $courier->zone_id);
                    });
                })
                ->orderBy('created_at')
                ->first();
        }

        if ($delivery) {
            $delivery->update([
                'courier_id' => $courier->id,
                'status' => Delivery::STATUS_PENDING,
                'assigned_at' => now(),
                'push_attempt' => 0,
                'push_last_sent_at' => null,
            ]);
            $this->notifyCourier($courier, $delivery);
        }

        return $delivery;
    }

    protected function notifyCourier(Courier $courier, Delivery $delivery): void
    {
        $users = $courier->users()->get();
        if ($users->isEmpty()) {
            return;
        }

        $payload = [
            'title' => 'Nuevo pedido',
            'body' => 'Tienes un pedido pendiente por aceptar.',
            'url' => url('/courier'),
            'tag' => 'delivery-' . $delivery->id,
            'delivery_id' => $delivery->id,
        ];

        $service = app(PushNotificationService::class);
        $telegram = app(TelegramNotificationService::class);
        foreach ($users as $user) {
            $service->sendToUser($user, $payload);
            $telegram->sendToUser($user, $telegram->formatCourierAssignment($delivery));
        }

        // Re-push reminders so Android "rings" again even if the PWA is in background/closed.
        // This requires a running queue worker (QUEUE_CONNECTION=database is already set in env files).
        if (config('queue.default') !== 'sync') {
            SendCourierPushReminder::dispatch($delivery->id, 1)->delay(now()->addSeconds(8));
            SendCourierPushReminder::dispatch($delivery->id, 2)->delay(now()->addSeconds(16));
        }
    }
}
